{
  title: 'SalesLoft',
  connection: {
    fields: [
      {
        name: 'client_id',
        label: 'Client ID',
        optional: false,
        hint: 'To create client id, you need to register an Oauth app. ' \
        " click <a href='https://accounts.salesloft.com/" \
        "oauth/applications' target='_blank'>here</a> to create app."
      },
      {
        name: 'client_secret',
        label: 'Client secret',
        control_type: 'password',
        optional: false,
        hint: 'To create client secret, you need to register an Oauth app. ' \
        " click <a href='https://accounts.salesloft.com/" \
        "oauth/applications' target='_blank'>here</a> to create app."
      }
    ],
    authorization: {
      type: 'oauth2',
      authorization_url: lambda do |connection|
        'https://accounts.salesloft.com/oauth/authorize?' \
        "client_id=#{connection['client_id']}&response_type=code"
      end,

      acquire: lambda do |connection, auth_code, redirect_uri|
        response = post('https://accounts.salesloft.com/oauth/token').
                   payload(client_id: connection['client_id'],
                           client_secret: connection['client_secret'],
                           grant_type: 'authorization_code',
                           code: auth_code,
                           redirect_uri: redirect_uri).
                   request_format_www_form_urlencoded
        [response, nil, nil]
      end,
      refresh_on: [401, 403],
      refresh: lambda do |connection, refresh_token|
        post('https://accounts.salesloft.com/oauth/token').
          payload(client_id: connection['client_id'],
                  client_secret: connection['client_secret'],
                  grant_type: 'refresh_token',
                  refresh_token: refresh_token).
          request_format_www_form_urlencoded
      end,
      apply: lambda do |_connection, access_token|
        headers(Authorization: "Bearer #{access_token}")
      end
    },
    base_uri: lambda do |_connection|
      'https://api.salesloft.com/'
    end
  },
  test: lambda do |_connection|
    get('v2/me.json')
  end,
  methods: {
    run: lambda do |input|
      payload = input['fields']
      path =
        if %w[calls emails].include?(payload['object_name'])
          "activities/#{payload['object_name']}"
        elsif payload['object_name'] == 'call_instructions'
          "action_details/#{payload['object_name']}"
        else
          payload['object_name']
        end

      if input['action'] == 'search'
        {
          (payload['object_name'] || 'records') =>
          get("v2/#{path}.json", payload.except('object_name')).
            after_error_response(/.*/) do |_code, body, _header, message|
              error("#{message}: #{body}")
            end['data']
        }
      elsif input['action'] == 'get'
        get("v2/#{path}/#{payload.delete('id')}.json",
            payload.except('object_name')).
          after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end['data']
      elsif input['action'] == 'create'
        post("v2/#{path}.json", payload.except('object_name')).
          after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end['data']
      elsif input['action'] == 'update'
        put("v2/#{payload.delete('object_name')}/#{payload.delete('id')}.json",
            payload).after_error_response(/.*/) do |_code, body, _header, message|
              error("#{message}: #{body}")
            end['data']
      else
        { status: delete("v2/#{payload['object_name']}/#{payload['id']}.json").
          after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end&.presence || 'Success' }
      end
    end,
    build_custom_field: lambda do |input|
      get("/v2/custom_fields.json?field_type=#{input}")['data']&.
      map do |field|
        {
          name: field['name'],
          type: (field['value_type'] == 'date' ? 'date' : 'string'),
          control_type: (field['value_type'] == 'date' ? 'date' : 'text')
        }
      end
    end,
    build_crm_activity_field: lambda do |input|
      get("/v2/crm_activity_fields.json?field_type=#{input}")['data']&.
      map do |field|
        {
          name: field['name'],
          type: (field['value_type'] == 'date' ? 'date' : 'string'),
          control_type: (field['value_type'] == 'date' ? 'date' : 'text')
        }
      end
    end,
    search_schema: lambda do |input|
      standard_fields = [
        { name: 'per_page', label: 'Page size', type: 'integer',
          control_type: 'integer', sticky: true, default: '25',
          hint: 'How many records to show per page in the range [1, 100].' \
          ' Defaults to 25' },
        { name: 'page', type: 'integer', control_type: 'integer',
          sticky: true, default: '1',
          hint: 'The current page to fetch results from. Defaults to 1' },
        { name: 'sort_by', control_type: 'select', sticky: true,
          pick_list: 'sort_fields',
          pick_list_params: { object: 'object_name' },
          hint: 'Field to sort the records with.',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'sort_by',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Sort by',
            toggle_hint: 'Use custom value',
            hint: 'E.g. created_at'
          } },
        { name: 'sort_direction', control_type: 'select', sticky: true,
          pick_list: 'sort_directions',
          hint: 'Direction to sort the records in. Leave blank to sort '\
          'records in descending order.',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'sort_direction',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Sort direction',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: ASC, DESC'
          } },
        { name: 'updated_at', label: 'Updated date', type: 'object',
          properties: [
            { name: 'date_updated', label: 'Date', type: 'date_time',
              sticky: true, hint: '<b>Filter operation</b> parameter is '\
              'required when <b>Date updated</b> is selected' },
            { name: 'operation', label: 'Filter operation',
              control_type: 'select', sticky: true,
              pick_list: 'operators',
              toggle_hint: 'Select operator',
              toggle_field: {
                name: 'operation',
                type: 'string',
                optional: true,
                control_type: 'text',
                label: 'Filter operation',
                toggle_hint: 'Use custom value',
                hint: 'Allowed values are: gt - Greater than, gte - Greater ' \
                'than or equal, lt - Less than, lte - Less than or equal '
              } }
          ] }
      ]
      case input
      when 'cadences'
        [
          { name: 'ids', label: 'IDs of cadences to fetch', sticky: true,
            hint: 'IDs of cadences to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing values '\
            'separated by comma' },
          { name: 'owned_by_guid', sticky: true, label: 'Owned by GUID',
            hint: 'Filters cadences by the owner\'s GUID. Multiple owner '\
            'GUIDs can be applied by providing values separated by comma' },
          { name: 'team_cadence',
            type: 'boolean',
            control_type: 'checkbox',
            sticky: true,
            hint: 'Filters cadences by whether they are a team ' \
            'cadence or not',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'team_cadence',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Team cadence',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'shared',
            type: 'boolean',
            control_type: 'checkbox',
            sticky: true,
            hint: 'Filters cadences by whether they are shared',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'shared',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Shared',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'people_addable',
            type: 'boolean',
            control_type: 'checkbox',
            sticky: true,
            hint: 'Filters cadences by whether they are able to have ' \
            'people added to them',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'people_addable',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'People addable',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'cadence_memberships'
        [
          { name: 'ids', label: 'IDs of cadence memberships to fetch',
            sticky: true,
            hint: 'IDs of cadence memberships to fetch. If a record can\'t ' \
            'be found, that record won\'t be returned and your request will be'\
            ' successful.<br/>Multiple IDs can be applied by providing values '\
            'separated by comma' },
          { name: 'person_id', type: 'integer', control_type: 'integer',
            sticky: true,
            hint: 'ID of the person to find cadence memberships for' },
          { name: 'cadence_id', type: 'integer', control_type: 'integer',
            sticky: true,
            hint: 'ID of the cadence to find cadence memberships for' },
          { name: 'currently_on_cadence', type: 'boolean',
            hint: 'If true, return only cadence memberships for people '\
            'currently on cadences. Else, return cadence memberships for '\
            'people who have been removed from or have completed a cadence.',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'currently_on_cadence',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Currently on cadence',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'people'
        [
          { name: 'ids', label: 'IDs of people to fetch', sticky: true,
            hint: 'IDs of people to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'owned_by_guid', sticky: true, label: 'Owned by GUID',
            hint: 'Filters people by the owner\'s GUID. Multiple owner '\
            'GUIDs can be applied by providing values separated by comma' },
          { name: 'person_stage_id', hint: 'Includes people that have a ' \
            'given person_stage. Multiple person stage IDs can be applied '\
            'by providing values separated by comma.' },
          { name: 'crm_id', label: 'CRM ID', sticky: true,
            hint: 'Filters people by crm_id. Multiple CRM IDs can be '\
            'applied by providing values separated by comma' },
          { name: 'account_id', hint: 'Filters people by the account they '\
            'are linked to. Multiple account IDs can be applied by '\
            'providing values separated by comma' },
          { name: 'email_addresses', hint: 'Provide comma separated list' \
            ' of emails for searching more than one person' },
          { name: 'do_not_contact', type: 'boolean',
            hint: 'Whether or not this person has opted out of all' \
            ' communication',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'do_not_contact',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Do not contact',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'can_email', type: 'boolean',
            hint: 'Includes people that can be emailed',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'can_email',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Can email',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'can_call', type: 'boolean',
            hint: 'Includes people that can be called',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'can_call',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Can call',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'emails'
        [
          { name: 'ids', label: 'IDs of emails to fetch', sticky: true,
            hint: 'IDs of emails to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'crm_activity_id', label: 'CRM activity ID', sticky: true,
            hint: 'Filters emails by crm_activity_id. Multiple CRM '\
            ' activity IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'action_id', sticky: true,
            hint: 'Filters emails by action_id. Multiple CRM action IDs ' \
            'can be applied by providing values separated by comma' },
          { name: 'bounced', type: 'boolean', sticky: true,
            hint: 'Filters emails by whether they have bounced or not',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'bounced',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Bounced',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'call_data_records'
        [
          { name: 'ids', label: 'IDs of call data records to fetch',
            sticky: true,
            hint: 'IDs of call data records to fetch. If a record can\'t ' \
            'be found, that record won\'t be returned and your request ' \
            'will be successful.<br/>Multiple IDs can be applied by '\
            'providing values separated by comma' },
          { name: 'user_guid', label: 'User GUID', sticky: true,
            hint: 'Provide comma separated GUIDs. Filters list to only '\
            'include GUIDs' },
          { name: 'created_at', type: 'object', properties: [
            { name: 'date_created', type: 'date_time', sticky: true,
              hint: '<b>Filter operation</b> parameter is required when ' \
              '<b>Date created</b> is selected' },
            { name: 'operation', label: 'Filter operation',
              sticky: true, control_type: 'select',
              pick_list: 'operators',
              toggle_hint: 'Select operator',
              toggle_field: {
                name: 'operation',
                type: 'string',
                optional: true,
                control_type: 'text',
                label: 'Filter operation',
                toggle_hint: 'Use custom value',
                hint: 'Allowed values are: gt - Greater than, gte - ' \
                'Gre than equal, lt - Less than, lte - Less than or equal '
              } }
          ] },
          { name: 'has_call', type: 'boolean', hint: 'Return only call '\
            'data records which have or do not have a call logged for them',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'has_call',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Has call',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'accounts'
        [
          { name: 'ids', label: 'IDs of accounts to fetch', sticky: true,
            hint: 'IDs of accounts to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'crm_id', label: 'CRM ID', sticky: true,
            hint: 'Filters account by crm_id. Multiple CRM IDs can be '\
            'applied by providing values separated by comma' },
          { name: 'accounts_staged_id', hint: 'Filters accounts by '\
            'account_stage_id. Multiple account_stage_ids can be applied'\
            ' providing values separated by comma.' },
          { name: 'name', sticky: true,
            hint: 'Names of accounts to fetch. Name matches'\
            ' are exact and case sensitive. Multiple names can be applied' \
            ' by providing values separated by comma.' },
          { name: 'tag', sticky: true,
            hint: 'Filters accounts by the tags applied to '\
            'the account. Multiple tags can be applied by providing '\
            'values separated by comma.' },
          { name: 'domain', sticky: true,
            hint: 'Domain of the accounts to fetch. '\
            'Domains are unique and lowercase' },
          { name: 'archived', type: 'boolean',
            hint: 'Returns only accounts where archived_at is not null if '\
            'this field is true. Returns only accounts where archived_at '\
            'is null if this field is false. ',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'archived',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Archived',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'account_tiers'
        [
          { name: 'ids', label: 'IDs of account tiers to fetch', sticky: true,
            hint: 'IDs of account tiers to fetch. If a record can\'t ' \
            'be found, that record won\'t be returned and your request ' \
            'will be successful.<br/>Multiple IDs can be applied by '\
            'providing values separated by comma' },
          { name: 'name', sticky: true,
            hint: 'Filters Account Tiers by name. Multiple '\
          'names can be applied by providing values separated by comma.' }
        ].concat(standard_fields.ignored('updated_at'))
      when 'actions'
        [
          { name: 'ids', label: 'IDs of actions to fetch', sticky: true,
            hint: 'IDs of actions to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'step_id', type: 'integer', control_type: 'integer',
            sticky: true, hint: 'Fetch actions by step ID' },
          { name: 'user_guid', label: 'User GUID', sticky: true,
            hint: 'Filters actions by the user\'s GUID. Multiple user'\
            ' GUIDs can be applied by providing values separated by '\
            'comma. The user must be a team admin to filter other '\
            'users\' actions' },
          { name: 'person_id', sticky: true,
            hint: 'Filters actions by <b>person_id</b>. Multiple person '\
            'IDs can be applied by providing values separated by  comma.' },
          { name: 'cadence_id', sticky: true,
            hint: 'Filters actions by <b>cadence_id</b>. Multiple cadence '\
            'IDs can be applied by providing values separated by  comma.' },
          { name: 'multitouch_group_id', sticky: true,
            hint: 'Filters actions by <b>multitouch_group_id</b>. Multiple multitouch_group '\
            'IDs can be applied by providing values separated by  comma.' },
          { name: 'person_id', sticky: true,
            hint: 'Filters actions by <b>person_id</b>. Multiple person '\
            'IDs can be applied by providing values separated by  comma.' },
          { name: 'type', sticky: true, hint: 'Filter actions by type' },
          { name: 'due_on', type: 'object', properties: [
            { name: 'date_due', type: 'date_time', sticky: true,
              hint: '<b>Filter operation</b> parameter is required when '\
              '<b>Date due</b> is selected' },
            { name: 'operation', control_type: 'select',
              sticky: true,
              pick_list: 'operators',
              toggle_hint: 'Select operator',
              toggle_field: {
                name: 'operation',
                type: 'string',
                optional: true,
                control_type: 'text',
                label: 'Filter operation',
                toggle_hint: 'Use custom value',
                hint: 'Allowed values are: gt - Greater than, gte - '\
                'Greater than or equal, lt - Less than, lte - Less ' \
                'than or equal '
              } }
          ] }
        ].concat(standard_fields.ignored('updated_at'))
      when 'call_instructions'
        [
          { name: 'ids', label: 'IDs of call instructions to fetch',
            sticky: true,
            hint: 'IDs of call instructions to fetch. If a record can\'t be ' \
            'found, that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' }
        ].concat(standard_fields.ignored('updated_at'))
      when 'call_sentiments', 'call_dispositions'
        [
          { name: 'ids', sticky: true,
            label: "IDs of #{input.labelize.downcase} to fetch",
            hint: 'IDs of call sentiments to fetch. If a record can\'t be ' \
            'found, that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' }
        ].concat(standard_fields.ignored('updated_at'))
      when 'custom_fields'
        [
          { name: 'ids', label: 'IDs of custom fields to fetch', sticky: true,
            hint: 'IDs of custom fields to fetch. If a record can\'t be ' \
            'found, that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'field_type', control_type: 'select',
            sticky: true,
            pick_list: 'field_types',
            toggle_hint: 'Select from list',
            hint: 'Type of field to fetch',
            toggle_field: {
              name: 'field_type',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Field type',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: <b>person</b>, <b>company</b>,'\
               ' <b>opportunity</b>'
            } }
        ].concat(standard_fields.ignored('updated_at'))
      when 'email_template_attachments'
        [
          { name: 'ids', label: 'IDs of email template attachments to fetch',
            sticky: true,
            hint: 'IDs of email template attachments to fetch. If a record ' \
            'can\'t be found, that record won\'t be returned and your '\
            'request will be successful.<br/>Multiple IDs can be applied '\
            'by providing values separated by comma' },
          { name: 'email_template_id', sticky: true,
            hint: 'Filters email template attachments by email '\
           'template IDs. Multiple IDs can be applied by providing values'\
           ' separated by  comma.' }
        ].concat(standard_fields.only('per_page', 'page'))
      when 'email_templates'
        [
          { name: 'ids', label: 'IDs of email templates to fetch',
            sticky: true,
            hint: 'IDs of email templates to fetch. If a record ' \
            'can\'t be found, that record won\'t be returned and your '\
            'request will be successful.<br/>Multiple IDs can be applied '\
            'by providing values separated by comma' },
          { name: 'group_id', sticky: true,
            hint: 'Filters email templates by groups applied to the '\
            'template by group ID. Not to exceed <b>500 IDs</b>. Returns '\
            'templates that are assigned to any of the group IDs. '\
            'Multiple IDs can be applied by providing values separated ' \
            'by comma.' },
          { name: 'tag_ids', label: 'Tag IDs', sticky: true,
            hint: 'Filters email templates by tags applied to the '\
            'template by tag ID, not to exceed <b>100 IDs</b>. Multiple IDs '\
            'can be applied by providing values separated by comma.' },
          { name: 'tag', hint: 'Filters email templates by tags applied'\
            ' to the template, not to exceed <b>100 tags</b>. Multiple tags'\
            ' can be applied by providing values separated by comma.' },
          { name: 'search', sticky: true,
            hint: 'Filters email templates by  title or subject' },
          { name: 'filter_by_owner', type: 'boolean',
            hint: 'Filters email templates by current authenticated user',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'filter_by_owner',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Filter by owner',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'linked_to_team_template', type: 'boolean',
            hint: 'Filters email templates by whether they are linked to a'\
            ' team template or not.',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'linked_to_team_template',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Linked to team template',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'include_cadence_templates', type: 'boolean',
            hint: 'Filters email templates based on whether or not the '\
            'template has been used on a cadence',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'include_cadence_templates',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Include cadence templates',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'include_archived_templates', type: 'boolean',
            hint: 'Filters email templates to include archived templates'\
            ' or not',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'include_archived_templates',
              type: 'string',
              control_type: 'text',
              label: 'Include archived templates',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'imports'
        [
          { name: 'ids', label: 'IDs of imports to fetch', sticky: true,
            hint: 'IDs of imports to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' },
          { name: 'user_ids', label: 'User IDs', sticky: true,
            hint: 'ID of users to fetch imports for. Using this filter '\
            'will return an empty array for non-admin users who request '\
            'other user\'s imports. Multiple IDs can be applied by '\
            'providing values separated by comma.' }
        ].concat(standard_fields.ignored('updated_at'))
      when 'notes'
        [
          { name: 'ids', label: 'IDs of notes to fetch', sticky: true,
            hint: 'IDs of notes to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' },
          { name: 'associated_with_type', control_type: 'select',
            sticky: true,
            pick_list: 'associated_types',
            toggle_hint: 'Select from list',
            hint: 'Case insensitive type of item with which the note' \
            ' is associated',
            toggle_field: {
              name: 'associated_with_type',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Associated with type',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: <b>person</b>, <b>account</b>'
            } },
          { name: 'associated_with_id', type: 'integer',
            control_type: 'integer', sticky: true,
            hint: 'ID of the item with which the note is associated. The '\
            '<b>associated_with_type</b> must also be present if this '\
            'parameter is used' }
        ].concat(standard_fields)
      when 'person_stages'
        [
          { name: 'ids', label: 'IDs of person stages to fetch', sticky: true,
            hint: 'IDs of person stages to fetch. If a record can\'t be found,'\
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' }
        ].concat(standard_fields.ignored('updated_at'))
      when 'saved_list_views'
        [
          { name: 'ids', label: 'IDs of saved list views to fetch',
            sticky: true,
            hint: 'IDs of saved list views to fetch. If a record can\'t be ' \
            'found, that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' },
          { name: 'view', control_type: 'select',
            sticky: true,
            pick_list: 'view_types',
            toggle_hint: 'Select from list',
            hint: 'Type of saved list view to fetch',
            toggle_field: {
              name: 'view',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'View',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: <b>people</b>, <b>companies</b>'
            } }
        ].concat(standard_fields.ignored('updated_at'))
      when 'steps'
        [
          { name: 'ids', label: 'IDs of steps to fetch', sticky: true,
            hint: 'IDs of steps to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' },
          { name: 'cadence_id', type: 'integer', control_type: 'integer',
            sticky: true, hint: 'Filter by cadence ID' },
          { name: 'type', control_type: 'select',
            sticky: true,
            pick_list: 'step_types',
            toggle_hint: 'Select from list',
            hint: 'Filter by step type',
            toggle_field: {
              name: 'type',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Step type',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: <b>email</b>, <b>phone</b>, '\
              '<b>integration</b>, <b>other</b>'
            } },
          { name: 'has_due_actions', sticky: true, type: 'boolean',
            hint: 'Filter by whether a step has due actions',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'has_due_actions',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Has due actions',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields.ignored('updated_at'))
      when 'successes'
        [
          { name: 'ids', label: 'IDs of successes to fetch', sticky: true,
            hint: 'IDs of successes to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' },
          { name: 'person_id', sticky: true,
            hint: 'Filters successes by person_id. Multiple person IDs can'\
            ' be applied by providing values separated by comma.' }
        ].concat(standard_fields)
      when 'tags'
        [
          { name: 'ids', label: 'IDs of tags to fetch', sticky: true,
            hint: 'IDs of tags to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing'\
            ' values separated by comma' },
          { name: 'search', label: 'Tag name', sticky: true,
            hint: 'Filters tags by name' }
        ].concat(standard_fields.ignored('updated_at'))
      when 'team_template_attachments'
        [
          { name: 'ids', label: 'IDs of team template attachments to fetch',
            sticky: true,
            hint: 'IDs of team template attachments to fetch. If a record '\
            'can\'t be found, that record won\'t be returned and your '\
            'request will be successful.<br/>Multiple IDs can be applied '\
            'by providing values separated by comma' },
          { name: 'team_template_id', sticky: true,
            hint: 'Filters template attachments by team template IDs. ' \
           'Multiple IDs can be applied by providing values separated '\
           'by comma' }
        ].concat(standard_fields.only('per_page', 'page'))
      when 'team_templates'
        [
          { name: 'ids', label: 'IDs of team templates to fetch',
            sticky: true,
            hint: 'IDs of team templates to fetch. If a record '\
            'can\'t be found, that record won\'t be returned and your '\
            'request will be successful.<br/>Multiple IDs can be applied '\
            'by providing values separated by comma' },
          { name: 'tag_ids', label: 'Tag IDs', sticky: true,
            hint: 'Filters team templates by tags applied to the template '\
            'by tag ID, not to exceed 100 IDs. Multiple IDs can be'\
            ' applied by providing values separated by comma.' },
          { name: 'tag', sticky: true,
            hint: 'Filters team templates by tags applied to the template,'\
            ' not to exceed 100 tags. Multiple tags can be'\
            ' applied by providing values separated by comma.' },
          { name: 'search', sticky: true,
            hint: 'Filters team templates by title or subject' },
          { name: 'include_archived_templates', type: 'boolean',
            hint: 'Filters team templates to include archived templates'\
            ' or not',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'include_archived_templates',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Include archived templates',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields)
      when 'users'
        [
          { name: 'ids', label: 'IDs of users to fetch', sticky: true,
            hint: 'IDs of users to fetch. If a record can\'t be found,' \
            ' that record won\'t be returned and your request will be ' \
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' },
          { name: 'guid', label: 'GUID',
            sticky: true,
            hint: 'Filters list to only include guids. Multiple IDs can be ' \
            'applied by providing values separated by comma.' },
          { name: 'group_id', sticky: true,
            hint: 'Filters users by group ID. Multiple ' \
            'IDs can be applied by providing values separated by comma.' },
          { name: 'role_id', label: 'Role ID', sticky: true,
            hint: 'Filters users by role_id.' },
          { name: 'search', sticky: true,
            hint: 'Space-separated list of keywords used to find ' \
            'case-insensitive substring matches against First Name, ' \
            'Last Name, or Email' },
          { name: 'active', type: 'boolean',
            hint: 'Filters users based on active attribute.',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'active',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Active',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'visible_only', type: 'boolean',
            hint: 'When true, only shows users that are actionable based on ' \
            "the team's privacy settings. When false, shows all users on the " \
            "team, even if you can't take action on that user. " \
            'Deactivated users are also included when false.',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'visible_only',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Visible only',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'include_paging', type: 'boolean',
            hint: 'Whether to include total_pages and total_count in ' \
            'the metadata. Defaults to false',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'include_paging',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Include paging',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ].concat(standard_fields.ignored('updated_at'))
      else
        [
          { name: 'ids', label: 'IDs of records to fetch', sticky: true,
            hint: 'IDs of records to fetch. If a record can\'t be found,'\
            ' that record won\'t be returned and your request will be '\
            'successful.<br/>Multiple IDs can be applied by providing '\
            'values separated by comma' }
        ].concat(standard_fields)
      end
    end,
    create_schema: lambda do |input|
      case input
      when 'cadence_memberships'
        [
          { name: 'person_id', type: 'integer',
            control_type: 'integer', optional: false,
            hint: 'ID of the person to create a cadence membership for' },
          { name: 'cadence_id',
            control_type: 'select',
            pick_list: 'cadences',
            optional: false,
            label: 'Cadence',
            hint: 'ID of the cadence to create a cadence membership for',
            toggle_hint: 'Select cadence',
            toggle_field: {
              name: 'cadence_id',
              type: 'integer',
              control_type: 'integer',
              label: 'Cadence ID',
              toggle_hint: 'Use cadence ID',
              hint: 'ID of the cadence to create a cadence membership for'
            } },
          { name: 'user_id', type: 'integer',
            sticky: true, control_type: 'integer',
            hint: 'ID of the user to create a cadence membership for. The ' \
            'associated cadence must be owned by the user, or it must be' \
            ' a team cadence' }
        ]
      when 'calls'
        crm_params = call('build_crm_activity_field', 'call')
        [
          { name: 'person_id', optional: false, control_type: 'integer',
            hint: 'The ID of the person whom this call will be logged for' },
          { name: 'to', sticky: true,
            hint: 'The phone number that was called' },
          { name: 'duration', type: 'integer', sticky: true,
            hint: 'Length of the call in seconds' },
          { name: 'disposition', sticky: true,
            control_type: 'select',
            pick_list: 'call_dispositions',
            hint: 'Result of the call.',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'disposition',
              label: 'Disposition',
              type: 'string',
              control_type: 'text',
              optional: true,
              toggle_hint: 'Use Custom Values',
              hint: 'Result of the call.' } },
          { name: 'sentiment', sticky: true,
            control_type: 'select',
            pick_list: 'call_sentiments',
            hint: 'Outcome of the conversation',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'sentiment',
              label: 'Sentiment',
              type: 'string',
              control_type: 'text',
              optional: true,
              toggle_hint: 'Use Custom Values',
              hint: 'Outcome of the conversation' } },
          { name: 'notes', hint: 'Notes to log for the call.' },
          { name: 'user_guid', label: 'User GUID',
            hint: 'GUID of the user whom this call should be logged for. ' \
            'Defaults to the authenticated user.' },
          { name: 'action_id',
            hint: 'Action that this call is being logged for. Multiple IDs can'\
            ' be applied by providing values separated by comma.' },
          { name: 'crm_params', type: 'object', properties: crm_params },
          { name: 'linked_call_data_record_ids',
            label: 'Linked call data record IDs',
            hint: 'CallDataRecord associations that will become linked to the '\
            'created call. Any call data record that is used must not already '\
            'be linked to a call. Multiple IDs can be applied by providing'\
            ' values separated by comma.' }
        ]
      when 'people'
        custom_fields = call('build_custom_field', 'person')
        [
          { name: 'first_name', sticky: true },
          { name: 'last_name', sticky: true },
          { name: 'email_address', sticky: true },
          { name: 'secondary_email_address', sticky: true },
          { name: 'personal_email_address', sticky: true },
          { name: 'phone', sticky: true },
          { name: 'phone_extension', sticky: true },
          { name: 'home_phone', sticky: true },
          { name: 'mobile_phone', sticky: true },
          { name: 'linkedin_url', label: 'LinkedIn URL', sticky: true },
          { name: 'title', label: 'Job title', sticky: true },
          { name: 'city', sticky: true },
          { name: 'state', sticky: true },
          { name: 'country', sticky: true },
          { name: 'work_city', sticky: true,
            label: 'Work location - city' },
          { name: 'work_state', sticky: true,
            label: 'Work location - state' },
          { name: 'work_country', sticky: true,
            label: 'Work location - country' },
          { name: 'person_company_name', sticky: true,
            label: 'Company name' },
          { name: 'person_company_website', sticky: true,
            label: 'Company website' },
          { name: 'person_company_industry', hint: 'Company industry' },
          { name: 'job_seniority',
            control_type: 'select',
            pick_list: 'job_seniorities',
            toggle_hint: 'Select job seniority',
            toggle_field: {
              name: 'job_seniority',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Job seniority',
              toggle_hint: 'Use custom value',
              hint: 'The Job Seniority of a Person, ' \
              'must be one of director, executive, individual_contributor, ' \
              'manager, vice_president, unknown'
            } },
          { name: 'do_not_contact', type: 'boolean',
            hint: 'Whether or not this person has opted out of all' \
            ' communication',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'do_not_contact',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Do not contact',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'locale', hint: 'Time locale of the person' },
          { name: 'personal_website', hint: 'The website of this person' },
          { name: 'twitter_handle', hint: 'The twitter handle of this person' },
          { name: 'tags', hint: 'All tags applied to this person. Multiple '\
            'tags can be applied by providing values separated by comma' },
          { name: 'contact_restrictions', hint: 'Specific methods of '\
          'communication to prevent for this person. Multiple restrictions'\
          'can be applied by providing values separated by comma' },
          { name: 'custom_fields', type: 'object', properties: custom_fields },
          { name: 'account_id', type: 'integer', control_type: 'integer' },
          { name: 'owner_id', type: 'integer', control_type: 'integer',
            hint: 'ID of the User that owns this person' },
          { name: 'import_id', type: 'integer', hint: 'ID of the Import this' \
            ' person is a part of. A person can be part of multiple imports,' \
            ' but this ID will always be the most recent Import' },
          { name: 'person_stage_id',
            control_type: 'select',
            pick_list: 'person_stages',
            label: 'Person stage',
            hint: 'ID of the PersonStage of this person',
            toggle_hint: 'Select person stage',
            toggle_field: {
              name: 'person_stage_id',
              type: 'integer',
              optional: true,
              control_type: 'integer',
              label: 'Person stage ID',
              toggle_hint: 'Use person stage ID',
              hint: 'ID of the PersonStage of this person'
            } },
          { name: 'crm_id', label: 'CRM ID',
            hint: 'ID of the person in your external CRM. You must ' \
            'provide a crm_id_type if this is included.' },
          { name: 'crm_id_type', label: 'CRM ID type',
            pick_list: [
              %w[salesforce salesforce]
            ],
            label: 'Person stage',
            hint: 'The CRM that the provided crm_id is for.',
            toggle_hint: 'Select CRM ID type',
            toggle_field: {
              name: 'crm_id_type',
              type: 'integer',
              optional: true,
              control_type: 'integer',
              label: 'CRM ID type',
              toggle_hint: 'Use custom value',
              hint: 'The CRM that the provided crm_id is for.'
            } },
          { name: 'autotag_date', type: 'boolean',
            hint: 'Whether the date should be added to this person as a tag. '\
            'Default is <b>false</b>. The tag will be <b>Y-m-d</b> format',
            control_type: 'checkbox',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'do_not_contact',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Do not contact',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ]
      when 'accounts'
        custom_fields = call('build_custom_field', 'company')
        [
          { name: 'name', label: 'Account full name', optional: false },
          { name: 'domain', optional: false,
            hint: 'Website domain, not a fully qualified URI' },
          { name: 'conversational_name', sticky: true,
            hint: 'Conversational name of the Account' },
          { name: 'description', sticky: true },
          { name: 'phone', sticky: true,
            hint: 'Phone number without formatting.' },
          { name: 'website', sticky: true },
          { name: 'linkedin_url', label: 'LinkedIn URL', sticky: true },
          { name: 'twitter_handle', hint: 'Twitter handle, with @' },
          { name: 'street', hint: 'Street name and number' },
          { name: 'city' },
          { name: 'state' },
          { name: 'postal_code' },
          { name: 'country' },
          { name: 'locale', label: 'Time locale' },
          { name: 'industry' },
          { name: 'company_type', hint: 'Type of the Account\'s company' },
          { name: 'founded', hint: 'Date or year of founding' },
          { name: 'revenue_range', sticky: true,
            hint: 'Estimated revenue range' },
          { name: 'size', sticky: true,
            hint: 'Estimated number of people in employment' },
          { name: 'do_not_contact', control_type: 'checkbox',
            type: 'boolean', sticky: true,
            render_input: 'boolean_conversion',
            parse_output: 'boolean_conversion',
            hint: 'Whether this company can not be contacted. Values are '\
            'either true or false. Setting this to true will remove all ' \
            'associated people from all active communications',
            toggle_hint: 'Select from option list',
            toggle_field: {
              name: 'do_not_contact',
              label: 'Do not contact',
              type: 'string',
              optional: true,
              control_type: 'text',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } },
          { name: 'custom_fields', type: 'object', properties: custom_fields },
          { name: 'tags', hint: 'All tags applied to this Account. Multiple '\
            'tags can be applied by providing values separated by comma.' },
          { name: 'owner_id', type: 'integer', control_type: 'integer',
            hint: 'ID of the User that owns this Account' },
          { name: 'company_stage_id', type: 'integer', control_type: 'integer',
            hint: 'ID of the CompanyStage assigned to this Account' }
        ]
      when 'custom_fields'
        [
          { name: 'name', label: 'Custom field name', optional: false,
            hint: '<b>Note:</b> The name of the custom field should not '\
          'contain special characters.' },
          { name: 'field_type', control_type: 'select',
            sticky: true,
            pick_list: 'field_types',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'field_type',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Field type',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: <b>person</b>, <b>company</b>,'\
              ' <b>opportunity</b>'
            } }
        ]
      when 'imports'
        [
          { name: 'user_id', control_type: 'integer', type: 'integer',
            sticky: true, hint: 'ID of the User that owns this Import' },
          { name: 'name', label: 'Import name', sticky: true }
        ]
      when 'notes'
        [
          { name: 'content', optional: false },
          { name: 'associated_with_type', control_type: 'select',
            optional: false,
            pick_list: 'associated_types',
            toggle_hint: 'Select from list',
            hint: 'Case insensitive type of item with which the note' \
            ' is associated',
            toggle_field: {
              name: 'associated_with_type',
              type: 'string',
              control_type: 'text',
              label: 'Associated with type',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: <b>person</b>, <b>account</b>'
            } },
          { name: 'associated_with_id', type: 'integer',
            control_type: 'integer', optional: false,
            hint: 'ID of the item with which the note is associated. The '\
            '<b>associated_with_type</b> must also be present if this '\
            'parameter is used' },
          { name: 'skip_crm_sync', label: 'Skip CRM sync', type: 'boolean',
            control_type: 'checkbox',
            hint: 'Indicates if the CRM sync should be skipped. No ' \
            'syncing will occur if true',
            toggle_hint: 'Select from list',
            toggle_field: {
              name: 'skip_crm_sync',
              type: 'string',
              optional: true,
              control_type: 'text',
              label: 'Skip CRM sync',
              hint: 'Allowed values are <b>true, false</b>',
              toggle_hint: 'Use custom value. '
            } },
          { name: 'call_id', control_type: 'integer', type: 'integer',
            hint: 'ID of the call with which the note is associated. The call' \
            ' cannot already have a note' },
          { name: 'subject', sticky: true, default: 'Note',
            hint: 'The subject of the note\'s CRM activity,' \
            ' defaults to <b>Note</b>' },
          { name: 'user_guid', label: 'User GUID',
            hint: 'The user to create the note for. Only team admins may '\
            'create notes on behalf of other users. Defaults to the '\
            'requesting user' }
        ]
      when 'person_stages'
        [{ name: 'name', optional: false, hint: 'The name of the new stage' }]
      when 'saved_list_views'
        call('saved_list_view_schema').ignored('id').required('name')
      when 'activities'
        [{ name: 'action_id', sticky: true,
           hint: 'Action that is being completed. This will'\
          ' validate that the action is still valid before completed it. The' \
          ' same action can never be successfully passed twice to this '\
          'endpoint. The action must have a type of integration.' }]
      else
        [{ name: 'action_id', sticky: true,
           hint: 'Action that is being marked ongoing. This'\
          ' will validate that the action is still valid before modifying it.'\
          ' Ongoing actions can not be marked ongoing.' }]
      end
    end,
    update_schema: lambda do |input|
      case input
      when 'accounts'
        [
          { name: 'id', label: 'Account ID' },
          { name: 'archived', control_type: 'checkbox',
            type: 'boolean',
            render_input: 'boolean_conversion',
            parse_output: 'boolean_conversion',
            toggle_hint: 'Select from option list',
            toggle_field: {
              name: 'archived',
              label: 'Archived',
              type: 'string',
              optional: true,
              control_type: 'text',
              toggle_hint: 'Use custom value',
              hint: 'Allowed values are: true, false'
            } }
        ]&.concat(call('create_schema', input))&.compact&.required('id')
      when 'custom_fields'
        call('custom_field_schema').only('name', 'id')
      when 'imports'
        [{ name: 'id', label: 'Import ID' }]&.
        concat(call('create_schema', input))&.compact&.required('id')
      when 'notes'
        [{ name: 'id', label: 'Note ID' }]&.
        concat(call('create_schema', input).only('content', 'call_id'))&.
        compact&.required('id')
      when 'people'
        [{ name: 'id', label: 'Person ID' }]&.
        concat(call('create_schema', input))&.compact&.
          ignored('autotag_date')&.required('id')
      when 'person_stages'
        [{ name: 'id', label: 'Stage ID' }]&.
        concat(call('create_schema', input))&.compact&.required('id')
      else
        call('saved_list_view_schema').ignored('view')
      end
    end,
    cadence_schema: lambda do
      [
        { name: 'id', label: 'Cadence ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'archived_at', type: 'date_time' },
        { name: 'team_cadence',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether this cadence is a team cadence. A team cadence ' \
          'is created by an admin and can be run by all users',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'team_cadence',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Team cadence',
            hint: 'Whether this cadence is a team cadence. A team cadence' \
            ' is created by an admin and can be run by all users',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'shared',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether this cadence is visible to team members (shared)',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'shared',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Shared',
            hint: 'Whether this cadence is visible to team members (shared)',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'remove_bounces_enabled',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether this cadence is configured to automatically ' \
          'remove people who have bounced',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'remove_bounces_enabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Remove bounces enabled',
            hint: 'Whether this cadence is configured to automatically ' \
            'remove people who have bounced',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'remove_replies_enabled',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether this cadence is configured to automatically ' \
          'remove people who have replied',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'remove_replies_enabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Remove replies enabled',
            hint: 'Whether this cadence is configured to automatically' \
            ' remove people who have replied',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'opt_out_link_included',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether this cadence is configured to include an ' \
          'opt-out link by default',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'opt_out_link_included',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Opt out link included',
            hint: 'Whether this cadence is configured to include an ' \
            'opt-out link by default',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'name', label: 'Cadence name' },
        { name: 'tags', hint: 'All tags applied to this cadence' },
        { name: 'creator', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'owner', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'bounced_stage', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'replied_stage', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'added_stage', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'finished_stage', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'counts', type: 'object', properties: [
          { name: 'cadence_people', type: 'integer' },
          { name: 'people_acted_on_count', type: 'integer' },
          { name: 'target_daily_people', type: 'integer' },
          { name: 'opportunities_created', type: 'integer' },
          { name: 'meetings_booked', type: 'integer' }
        ] }
      ]
    end,
    cadence_membership_schema: lambda do
      [
        { name: 'id', type: 'integer', label: 'Cadence membership ID' },
        { name: 'added_at', type: 'date_time' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'person_deleted',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether the associated person has since been deleted',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'person_deleted',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Person deleted',
            hint: 'Whether the associated person has since been deleted',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'current_state' },
        { name: 'currently_on_cadence' },
        { name: 'cadence', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'latest_action', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'counts', type: 'object', properties: [
          { name: 'views', type: 'integer' },
          { name: 'clicks', type: 'integer' },
          { name: 'replies', type: 'integer' },
          { name: 'calls', type: 'integer' },
          { name: 'sent_emails', type: 'integer' },
          { name: 'bounces', type: 'integer' }
        ] }
      ]
    end,
    call_schema: lambda do
      [
        { name: 'id', label: 'Call ID' },
        { name: 'to' },
        { name: 'duration', type: 'integer' },
        { name: 'sentiment' },
        { name: 'disposition' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'recordings', type: 'array', of: :object, properties: [
          { name: 'url', label: 'URL' },
          { name: 'recording' },
          { name: 'status' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'action', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'called_person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'crm_activity', label: 'CRM activity', type: 'object',
          properties: [
            { name: 'id', type: 'integer' },
            { name: '_href', label: 'HREF' }
          ] },
        { name: 'note', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'cadence', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'step', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    email_schema: lambda do
      [
        { name: 'id', label: 'Email ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'recipient_email_address', control_type: 'email' },
        { name: 'status' },
        { name: 'bounced', type: 'boolean' },
        { name: 'send_after', type: 'date_time' },
        { name: 'sent_at', type: 'date_time' },
        { name: 'view_tracking', type: 'boolean' },
        { name: 'click_tracking', type: 'boolean' },
        { name: 'headers', type: 'object', properties: [
          { name: 'cc', control_type: 'email' },
          { name: 'bcc', control_type: 'email' }
        ] },
        { name: 'counts', type: 'object', properties: [
          { name: 'clicks', type: 'integer' },
          { name: 'views', type: 'integer' },
          { name: 'replies', type: 'integer' },
          { name: 'unique_devices', type: 'integer' },
          { name: 'unique_locations', type: 'integer' },
          { name: 'attachments', type: 'integer' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'recipient', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'mailing', type: 'object', properties: [
          { name: 'id', type: 'integer' }
        ] },
        { name: 'action', type: 'object', properties: [
          { name: 'id', type: 'integer' }
        ] },
        { name: 'crm_activity', label: 'CRM activity', type: 'object',
          properties: [
            { name: 'id', type: 'integer' },
            { name: '_href', label: 'HREF' }
          ] },
        { name: 'cadence', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'step', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'email_template', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    crm_activity_schema: lambda do
      [
        { name: 'id', label: 'CRM activity ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'subject' },
        { name: 'description' },
        { name: 'crm_id', label: 'CRM ID' },
        { name: 'activity_type' },
        { name: 'error' },
        { name: 'person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    call_data_record_schema: lambda do
      [
        { name: 'id', label: 'Call data record ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'to' },
        { name: 'from' },
        { name: 'duration', type: 'integer' },
        { name: 'direction' },
        { name: 'status' },
        { name: 'call_type' },
        { name: 'call_uuid', label: 'Call UUID' },
        { name: 'recording', type: 'object', properties: [
          { name: 'url', label: 'URL' },
          { name: 'status' },
          { name: 'recording_status' }
        ] },
        { name: 'call', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'called_person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    account_schema: lambda do
      custom_fields = call('build_custom_field', 'company')
      [
        { name: 'id', label: 'Account ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'archived_at', type: 'date_time' },
        { name: 'name', label: 'Account full name' },
        { name: 'domain' },
        { name: 'conversational_name',
          hint: 'Conversational name of the Account' },
        { name: 'description' },
        { name: 'phone' },
        { name: 'website' },
        { name: 'linkedin_url', label: 'LinkedIn URL' },
        { name: 'twitter_handle' },
        { name: 'street' },
        { name: 'city' },
        { name: 'state' },
        { name: 'postal_code' },
        { name: 'country' },
        { name: 'locale', label: 'Time locale' },
        { name: 'industry' },
        { name: 'company_type' },
        { name: 'founded' },
        { name: 'revenue_range' },
        { name: 'size' },
        { name: 'crm_id', label: 'CRM ID' },
        { name: 'crm_url', label: 'CRM URL' },
        { name: 'crm_object_type', label: 'CRM object type' },
        { name: 'owner_crm_id', label: 'Owner CRM ID' },
        { name: 'last_contacted_at', type: 'date_time' },
        { name: 'last_contacted_type' },
        { name: 'do_not_contact', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'do_not_contact',
            label: 'Do not contact',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'custom_fields', type: 'object', properties: custom_fields },
        { name: 'tags' },
        { name: 'counts', type: 'object', properties: [
          { name: 'people', type: 'integer' }
        ] },
        { name: 'owner', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'creator', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'last_contacted_by', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'last_contacted_person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'company_stage', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'account_tier', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    person_schema: lambda do
      custom_fields = call('build_custom_field', 'person')
      [
        { name: 'id', type: 'integer', label: 'Person ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'last_contacted_at', type: 'date_time' },
        { name: 'last_replied_at', type: 'date_time' },
        { name: 'first_name' },
        { name: 'last_name' },
        { name: 'display_name' },
        { name: 'email_address' },
        { name: 'secondary_email_address' },
        { name: 'personal_email_address' },
        { name: 'phone' },
        { name: 'phone_extension' },
        { name: 'home_phone' },
        { name: 'mobile_phone' },
        { name: 'linkedin_url', label: 'LinkedIn URL' },
        { name: 'title', label: 'Position' },
        { name: 'city' },
        { name: 'state' },
        { name: 'country' },
        { name: 'work_city', label: 'Work location - city' },
        { name: 'work_state', label: 'Work location - state' },
        { name: 'work_country', label: 'Work location - country' },
        { name: 'crm_url', label: 'CRM URL',
          hint: 'CRM URL, currently support Salesforce.com only' },
        { name: 'crm_id', label: 'CRM ID',
          hint: 'CRM ID, currently support Salesforce.com only' },
        { name: 'crm_object_type', label: 'CRM object type',
          hint: 'CRM object type, currently support Salesforce.com only' },
        { name: 'owner_crm_id', label: 'Owner CRM ID',
          hint: 'Mapped owner field from the CRM,' \
          ' currently Salesforce.com only' },
        { name: 'person_company_name', label: 'Company name' },
        { name: 'person_company_website', label: 'Company website' },
        { name: 'person_company_industry', hint: 'Company industry' },
        { name: 'do_not_contact', type: 'boolean',
          hint: 'Whether or not this person has opted out of all' \
          ' communication',
          control_type: 'checkbox',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'do_not_contact',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Do not contact',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'bouncing',
          type: 'boolean',
          control_type: 'checkbox',
          hint: 'Whether this person\'s current email address has bounced',
          toggle_hint: 'Select from list',
          toggle_field: {
            name: 'bouncing',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Bouncing',
            toggle_hint: 'Use custom value. ' \
            'Allowed values are <b>true, false</b>'
          } },
        { name: 'locale', hint: 'Time locale of the person' },
        { name: 'personal_website', hint: 'The website of this person' },
        { name: 'twitter_handle', hint: 'The twitter handle of this person' },
        { name: 'last_contacted_type', hint: 'The type of the last touch' \
          ' to this person. Can be call, email, other' },
        { name: 'job_seniority',
          control_type: 'select',
          pick_list: 'job_seniorities',
          toggle_hint: 'Select job seniority',
          toggle_field: {
            name: 'job_seniority',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Job seniority',
            toggle_hint: 'Use custom value',
            hint: 'The Job Seniority of a Person, ' \
            'must be one of director, executive, individual_contributor, ' \
            'manager, vice_president, unknown'
          } },
        { name: 'custom_fields', type: 'object', properties: custom_fields },
        { name: 'tags', hint: 'All tags applied to this person' },
        { name: 'contact_restrictions',
          control_type: 'multiselect',
          pick_list: 'contact_restrictions',
          toggle_hint: 'Select contact restrictions',
          toggle_field: {
            name: 'contact_restrictions',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Contact restrictions',
            toggle_hint: 'Use custom values',
            hint: 'Specific methods of communication to prevent for this ' \
            'person. This will prevent individual execution of these' \
            ' communication types as well as automatically skip cadence ' \
            'steps of this communication type for this person in SalesLoft.' \
            ' Values currently accepted: call, email, message'
          },
          hint: 'Specific methods of communication to prevent for ' \
          'this person.' },
        { name: 'counts', type: 'object', properties: [
          { name: 'emails_sent', type: 'integer' },
          { name: 'emails_viewed', type: 'integer' },
          { name: 'emails_clicked', type: 'integer' },
          { name: 'emails_replied_to', type: 'integer' },
          { name: 'emails_bounced', type: 'integer' },
          { name: 'calls', type: 'integer' }
        ] },
        { name: 'account', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'owner', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'last_contacted_by', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'import', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'person_stage', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    note_schema: lambda do
      [
        { name: 'id', label: 'Note ID' },
        { name: 'content', hint: 'The content of the note' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'associated_with', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'call', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    person_stage_schema: lambda do
      [
        { name: 'id', label: 'Person stage ID' },
        { name: 'name', label: 'Person stage name' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'order', type: 'integer' }
      ]
    end,
    view_fields: lambda do
      [
        { name: 'owner', sticky: true, control_type: 'select',
          pick_list: 'owners',
          toggle_hint: 'Select job seniority',
          toggle_field: {
            name: 'owner',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Owner',
            toggle_hint: 'Use custom value'
          } },
        { name: 'job_seniority', sticky: true,
          control_type: 'select',
          pick_list: 'job_seniorities',
          toggle_hint: 'Select job seniority',
          toggle_field: {
            name: 'job_seniority',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Job seniority',
            toggle_hint: 'Use custom value',
            hint: 'The Job Seniority of a Person, ' \
            'must be one of director, executive, individual_contributor, ' \
            'manager, vice_president, unknown'
          } }
      ]
    end,
    saved_list_view_schema: lambda do
      view_params = call('view_fields')
      [
        { name: 'id', label: 'Saved list view ID', optional: false },
        { name: 'view', control_type: 'select',
          optional: false,
          pick_list: 'view_types',
          toggle_hint: 'Select from list',
          hint: 'Type of saved list view',
          toggle_field: {
            name: 'view',
            type: 'string',
            optional: false,
            control_type: 'text',
            label: 'View',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: <b>people</b>, <b>companies</b>'
          } },
        { name: 'name', label: 'Saved list view name' },
        { name: 'view_params', label: 'List of set filters',
          type: 'object', properties: view_params },
        { name: 'is_default', control_type: 'checkbox',
          type: 'boolean', sticky: true,
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'is_default',
            label: 'Is default',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } }
      ]
    end,
    account_stage_schema: lambda do
      [
        { name: 'id', label: 'Account stage ID' },
        { name: 'name', label: 'Account stage name' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'order', type: 'integer' }
      ]
    end,
    account_tier_schema: lambda do
      [
        { name: 'id', label: 'Account tier ID' },
        { name: 'name', label: 'Account tier name' },
        { name: 'order', type: 'integer' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' }
      ]
    end,
    action_schema: lambda do
      [
        { name: 'id', label: 'Action ID' },
        { name: 'due', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'due',
            label: 'Due',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'type' },
        { name: 'status' },
        { name: 'multitouch_group_id', type: 'integer' },
        { name: 'due_on', type: 'date_time' },
        { name: 'action_details', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'cadence', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'step', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    call_sentiment_schema: lambda do
      [
        { name: 'id', label: 'Call sentiment ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'name', label: 'Call sentiment name' }
      ]
    end,
    call_disposition_schema: lambda do
      [
        { name: 'id', label: 'Call disposition ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'name', label: 'Call disposition name' }
      ]
    end,
    custom_field_schema: lambda do
      [
        { name: 'id', label: 'Custom field ID', optional: false },
        { name: 'name', label: 'Custom field name', sticky: true,
          hint: '<b>Note:</b> The name of the custom field should not contain'\
        'special characters.' },
        { name: 'field_type', control_type: 'select',
          sticky: true,
          pick_list: 'field_types',
          toggle_hint: 'Select from list',
          hint: 'Type of field to fetch',
          toggle_field: {
            name: 'field_type',
            type: 'string',
            optional: true,
            control_type: 'text',
            label: 'Field type',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: <b>person</b>, <b>company</b>,'\
           ' <b>opportunity</b>'
          } },
        { name: 'value_type' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' }
      ]
    end,
    activity_schema: lambda do
      [{ name: 'updated_at', type: 'date_time' }]
    end,
    ongoing_action_schema: lambda do
      [
        { name: 'id', label: 'Ongoing action ID' },
        { name: 'due', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'due',
            label: 'Due',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'type' },
        { name: 'status' },
        { name: 'due_on', type: 'date_time' },
        { name: 'action_details', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    call_instruction_schema: lambda do
      [
        { name: 'id', label: 'Call instruction ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'instructions' }
      ]
    end,
    email_template_attachment_schema: lambda do
      [
        { name: 'id', label: 'Email template attachment ID' },
        { name: 'attachment_id' },
        { name: 'email_template', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'name', label: 'Attachment name' },
        { name: 'download_url', label: 'Download URL' },
        { name: 'attachment_file_size', type: 'integer' }
      ]
    end,
    email_template_schema: lambda do
      [
        { name: 'id', label: 'Email template ID' },
        { name: 'title' },
        { name: 'subject' },
        { name: 'body' },
        { name: 'body_preview' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'last_used_at', type: 'date_time' },
        { name: 'archived_at', type: 'date_time' },
        { name: 'shared', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'shared',
            label: 'Shared',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'open_tracking_enabled', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'open_tracking_enabled',
            label: 'Open tracking enabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'click_tracking_enabled', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'click_tracking_enabled',
            label: 'Click tracking enabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'cadence_template', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'cadence_template',
            label: 'Cadence template',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'counts', type: 'object', properties: [
          { name: 'sent_emails', type: 'integer' },
          { name: 'views', type: 'integer' },
          { name: 'clicks', type: 'integer' },
          { name: 'replies', type: 'integer' },
          { name: 'bounces', type: 'integer' }
        ] },
        { name: 'template_owner', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'team_template', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: '_links', type: 'object', properties: [
          { name: 'attachments' }
        ] },
        { name: 'tags' },
        { name: 'groups', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    import_schema: lambda do
      [
        { name: 'id', label: 'Import ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'name', label: 'Import name' },
        { name: 'current_people_count', type: 'integer' },
        { name: 'imported_people_count', type: 'integer' }
      ]
    end,
    step_schema: lambda do
      [
        { name: 'id', label: 'Step ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'disabled', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'disabled',
            label: 'Disabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'type' },
        { name: 'display_name' },
        { name: 'day', type: 'integer' },
        { name: 'step_number', type: 'integer' },
        { name: 'details', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'cadence', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] }
      ]
    end,
    success_schema: lambda do
      [
        { name: 'id', label: 'Success ID' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'succeeded_at', type: 'date_time' },
        { name: 'success_window_started_at', type: 'date_time' },
        { name: 'user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'person', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'latest_email', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'latest_call', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'latest_action', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'latest_cadence', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'counts', type: 'object', properties: [
          { name: 'total_emails', type: 'integer' },
          { name: 'total_calls', type: 'integer' },
          { name: 'total_other_touches', type: 'integer' }
        ] }
      ]
    end,
    tag_schema: lambda do
      [
        { name: 'id', label: 'Tag ID' },
        { name: 'name', type: 'Tag name' }
      ]
    end,
    team_template_attachment_schema: lambda do
      [
        { name: 'id', label: 'Team template attachment ID' },
        { name: 'attachment_id' },
        { name: 'team_template', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: 'name', label: 'Attachment name' },
        { name: 'download_url', label: 'Download URL' },
        { name: 'attachment_file_size', type: 'integer' }
      ]
    end,
    team_template_schema: lambda do
      [
        { name: 'id', label: 'Team template ID' },
        { name: 'title' },
        { name: 'subject' },
        { name: 'body' },
        { name: 'body_preview' },
        { name: 'created_at', type: 'date_time' },
        { name: 'updated_at', type: 'date_time' },
        { name: 'last_used_at', type: 'date_time' },
        { name: 'archived_at', type: 'date_time' },
        { name: 'last_modified_at', type: 'date_time' },
        { name: 'open_tracking_enabled', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'open_tracking_enabled',
            label: 'Open tracking enabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'click_tracking_enabled', control_type: 'checkbox',
          type: 'boolean',
          render_input: 'boolean_conversion',
          parse_output: 'boolean_conversion',
          toggle_hint: 'Select from option list',
          toggle_field: {
            name: 'click_tracking_enabled',
            label: 'Click tracking enabled',
            type: 'string',
            optional: true,
            control_type: 'text',
            toggle_hint: 'Use custom value',
            hint: 'Allowed values are: true, false'
          } },
        { name: 'counts', type: 'object', properties: [
          { name: 'sent_emails', type: 'integer' },
          { name: 'views', type: 'integer' },
          { name: 'clicks', type: 'integer' },
          { name: 'replies', type: 'integer' },
          { name: 'bounces', type: 'integer' }
        ] },
        { name: 'last_modified_user', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href', label: 'HREF' }
        ] },
        { name: '_links', type: 'object', properties: [
          { name: 'attachments' }
        ] },
        { name: 'tags' }
      ]
    end,
    user_schema: lambda do
      [
        { name: 'id', label: 'User ID', type: 'integer' },
        { name: 'guid', label: 'GUID', type: 'integer' },
        { name: 'created_at', label: 'Date created', type: 'date_time' },
        { name: 'updated_at', label: 'Date updated', type: 'date_time' },
        { name: 'name' },
        { name: 'first_name' },
        { name: 'last_name' },
        { name: 'active', type: 'boolean' },
        { name: 'time_zone' },
        { name: 'slack_username' },
        { name: 'twitter_handle' },
        { name: 'email' },
        { name: 'email_client_email_address' },
        { name: 'sending_email_address  ' },
        { name: 'from_address' },
        { name: 'full_email_address' },
        { name: 'bcc_email_address' },
        { name: 'email_signature' },
        { name: 'email_signature_type' },
        { name: 'email_signature_click_tracking_disabled', type: 'boolean' },
        { name: 'team_admin', type: 'boolean' },
        { name: 'local_dial_enabled', type: 'boolean' },
        { name: 'click_to_call_enabled', type: 'boolean' },
        { name: 'email_client_configured', type: 'boolean' },
        { name: 'crm_connected', label: 'CRM connected', type: 'boolean' },
        { name: 'external_feature_flags', type: 'object', properties: [
          { name: 'ma_enabled', type: 'boolean' },
          { name: 'ma_live_feed', type: 'boolean' },
          { name: 'hot_leads', type: 'boolean' }
        ] },
        { name: 'phone_client', type: 'object', properties: [
          { name: 'id', type: 'integer' }
        ] },
        { name: 'phone_number_assignment', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href' }
        ] },
        { name: 'group', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href' }
        ] },
        { name: 'team', type: 'object', properties: [
          { name: 'id', type: 'integer' },
          { name: '_href' }
        ] }
      ]
    end,
    get_object_fields: lambda do |input|
      case input
      when 'cadences'
        call('cadence_schema')
      when 'cadence_memberships'
        call('cadence_membership_schema')
      when 'people'
        call('person_schema')
      when 'emails'
        call('email_schema')
      when 'crm_activities'
        call('crm_activity_schema')
      when 'call_data_records'
        call('call_data_record_schema')
      when 'accounts'
        call('account_schema')
      when 'calls'
        call('call_schema')
      when 'notes'
        call('note_schema')
      when 'person_stages'
        call('person_stage_schema')
      when 'saved_list_views'
        call('saved_list_view_schema')
      when 'account_stages'
        call('account_stage_schema')
      when 'account_tiers'
        call('account_tier_schema')
      when 'actions'
        call('action_schema')
      when 'activities'
        call('activity_schema')
      when 'ongoing_actions'
        call('ongoing_action_schema')
      when 'call_instructions'
        call('call_instruction_schema')
      when 'call_dispositions'
        call('call_disposition_schema')
      when 'call_sentiments'
        call('call_sentiment_schema')
      when 'custom_fields'
        call('custom_field_schema')
      when 'email_template_attachments'
        call('email_template_attachment_schema')
      when 'email_templates'
        call('email_template_schema')
      when 'imports'
        call('import_schema')
      when 'steps'
        call('step_schema')
      when 'successes'
        call('success_schema')
      when 'tags'
        call('tag_schema')
      when 'team_template_attachments'
        call('team_template_attachment_schema')
      when 'team_templates'
        call('team_template_schema')
      when 'users'
        call('user_schema')
      else
        []
      end
    end,
    sample_record: lambda do |input|
      object = input.delete('object_name')
      path = if %w[calls emails].include?(object)
               "activities/#{object}"
             elsif object == 'call_instructions'
               "action_details/#{object}"
             else
               object
             end
      get("v2/#{path}.json?per_page=1").dig('data', 0) || {}
    end,
    ##############################################################
    # Helper methods                                             #
    ##############################################################
    # This method is for Custom action
    make_schema_builder_fields_sticky: lambda do |schema|
      schema.map do |field|
        if field['properties'].present?
          field['properties'] = call('make_schema_builder_fields_sticky',
                                     field['properties'])
        end
        field['sticky'] = true

        field
      end
    end,

    # Formats input/output schema to replace any special characters in name,
    # without changing other attributes (method required for custom action)
    format_schema: lambda do |input|
      input&.map do |field|
        if (props = field[:properties])
          field[:properties] = call('format_schema', props)
        elsif (props = field['properties'])
          field['properties'] = call('format_schema', props)
        end
        if (name = field[:name])
          field[:label] = field[:label].presence || name.labelize
          field[:name] = name.
                         gsub(/\W/) { |spl_chr| "__#{spl_chr.encode_hex}__" }
        elsif (name = field['name'])
          field['label'] = field['label'].presence || name.labelize
          field['name'] = name.
                          gsub(/\W/) { |spl_chr| "__#{spl_chr.encode_hex}__" }
        end

        field
      end
    end,

    # Formats payload to inject any special characters that previously removed
    format_payload: lambda do |payload|
      if payload.is_a?(Array)
        payload.map do |array_value|
          call('format_payload', array_value)
        end
      elsif payload.is_a?(Hash)
        payload.each_with_object({}) do |(key, value), hash|
          key = key.gsub(/__\w+__/) do |string|
            string.gsub(/__/, '').decode_hex.as_utf8
          end
          if value.is_a?(Array) || value.is_a?(Hash)
            value = call('format_payload', value)
          end
          hash[key] = value
        end
      end
    end,

    # Formats response to replace any special characters with valid strings
    # (method required for custom action)
    format_response: lambda do |response|
      response = response&.compact unless response.is_a?(String) || response
      if response.is_a?(Array)
        response.map do |array_value|
          call('format_response', array_value)
        end
      elsif response.is_a?(Hash)
        response.each_with_object({}) do |(key, value), hash|
          key = key.gsub(/\W/) { |spl_chr| "__#{spl_chr.encode_hex}__" }
          if value.is_a?(Array) || value.is_a?(Hash)
            value = call('format_response', value)
          end
          hash[key] = value
        end
      else
        response
      end
    end
  },
  object_definitions: {
    object_search: {
      fields: lambda do |_connection, config_fields|
        next [] if config_fields.blank?
        call('search_schema', config_fields['object_name'])
      end
    },
    object_create: {
      fields: lambda do |_connehgction, config_fields|
        next [] if config_fields.blank?
        call('create_schema', config_fields['object_name'])
      end
    },
    object_update: {
      fields: lambda do |_connehgction, config_fields|
        next [] if config_fields.blank?
        call('update_schema', config_fields['object_name'])
      end
    },
    object_get: {
      fields: lambda do |_connection, config_fields|
        next [] if config_fields.blank?
        if config_fields['object_name'] == 'team_templates'
          [
            { name: 'include_signature',
              type: 'boolean',
              control_type: 'checkbox',
              sticky: true,
              hint: 'Optionally will return the templates with the current '\
              'user\'s email signature',
              toggle_hint: 'Select from list',
              toggle_field: {
                name: 'include_signature',
                type: 'string',
                optional: true,
                control_type: 'text',
                label: 'Include signature',
                toggle_hint: 'Use custom value',
                hint: 'Allowed values are: true, false'
              } }
          ]
        else
          []
        end&.concat(
          [{ name: 'id', optional: false,
             hint: 'Provide the SalesLoft internal ID of the object ' \
            'you want to retrieve' }])&.compact
      end
    },
    object_output: {
      fields: lambda do |_connection, config_fields|
        next [] if config_fields.blank?
        call('get_object_fields', config_fields['object_name'])
      end
    },
    record_search_output: {
      fields: lambda do |_connection, config_fields|
        next [] if config_fields.blank?
        [
          { name: (config_fields&.[]('object_name')&.pluralize || 'records'),
            type: 'array', of: 'object',
            properties: call('get_object_fields', config_fields['object_name']) },
          { name: 'metadata', type: 'object',
            properties: [
              { name: 'paging', type: 'object',
                properties: [
                  { name: 'per_page', type: 'integer' },
                  { name: 'current_page', type: 'integer' },
                  { name: 'next_page' },
                  { name: 'prev_page' }
                ] }
            ] }
        ]
      end
    },
    custom_action_input: {
      fields: lambda do |_connection, config_fields|
        verb = config_fields['verb']
        input_schema = parse_json(config_fields.dig('input', 'schema') || '[]')
        data_props =
          input_schema.map do |field|
            if config_fields['request_type'] == 'multipart' &&
               field['binary_content'] == 'true'
              field['type'] = 'object'
              field['properties'] = [
                { name: 'file_content', optional: false },
                {
                  name: 'content_type',
                  default: 'text/plain',
                  sticky: true
                },
                { name: 'original_filename', sticky: true }
              ]
            end
            field
          end
        data_props = call('make_schema_builder_fields_sticky', data_props)
        input_data =
          if input_schema.present?
            if input_schema.dig(0, 'type') == 'array' &&
               input_schema.dig(0, 'details', 'fake_array')
              {
                name: 'data',
                type: 'array',
                of: 'object',
                properties: data_props.dig(0, 'properties')
              }
            else
              { name: 'data', type: 'object', properties: data_props }
            end
          end

        [
          {
            name: 'path',
            hint: 'Base URI is <b>' \
            'https://api.salesloft.com/' \
            '</b> - path will be appended to this URI. Use absolute URI to ' \
            'override this base URI.',
            optional: false
          },
          if %w[post put patch].include?(verb)
            {
              name: 'request_type',
              default: 'json',
              sticky: true,
              extends_schema: true,
              control_type: 'select',
              pick_list: [
                ['JSON request body', 'json'],
                ['URL encoded form', 'url_encoded_form'],
                ['Mutipart form', 'multipart'],
                ['Raw request body', 'raw']
              ]
            }
          end,
          {
            name: 'response_type',
            default: 'json',
            sticky: false,
            extends_schema: true,
            control_type: 'select',
            pick_list: [['JSON response', 'json'], ['Raw response', 'raw']]
          },
          if %w[get options delete].include?(verb)
            {
              name: 'input',
              label: 'Request URL parameters',
              sticky: true,
              add_field_label: 'Add URL parameter',
              control_type: 'form-schema-builder',
              type: 'object',
              properties: [
                {
                  name: 'schema',
                  sticky: input_schema.blank?,
                  extends_schema: true
                },
                input_data
              ].compact
            }
          else
            {
              name: 'input',
              label: 'Request body parameters',
              sticky: true,
              type: 'object',
              properties:
                if config_fields['request_type'] == 'raw'
                  [{
                    name: 'data',
                    sticky: true,
                    control_type: 'text-area',
                    type: 'string'
                  }]
                else
                  [
                    {
                      name: 'schema',
                      sticky: input_schema.blank?,
                      extends_schema: true,
                      schema_neutral: true,
                      control_type: 'schema-designer',
                      sample_data_type: 'json_input',
                      custom_properties:
                        if config_fields['request_type'] == 'multipart'
                          [{
                            name: 'binary_content',
                            label: 'File attachment',
                            default: false,
                            optional: true,
                            sticky: true,
                            render_input: 'boolean_conversion',
                            parse_output: 'boolean_conversion',
                            control_type: 'checkbox',
                            type: 'boolean'
                          }]
                        end
                    },
                    input_data
                  ].compact
                end
            }
          end,
          {
            name: 'request_headers',
            sticky: false,
            extends_schema: true,
            control_type: 'key_value',
            empty_list_title: 'Does this HTTP request require headers?',
            empty_list_text: 'Refer to the API documentation and add ' \
            'required headers to this HTTP request',
            item_label: 'Header',
            type: 'array',
            of: 'object',
            properties: [{ name: 'key' }, { name: 'value' }]
          },
          unless config_fields['response_type'] == 'raw'
            {
              name: 'output',
              label: 'Response body',
              sticky: true,
              extends_schema: true,
              schema_neutral: true,
              control_type: 'schema-designer',
              sample_data_type: 'json_input'
            }
          end,
          {
            name: 'response_headers',
            sticky: false,
            extends_schema: true,
            schema_neutral: true,
            control_type: 'schema-designer',
            sample_data_type: 'json_input'
          }
        ].compact
      end
    },
    custom_action_output: {
      fields: lambda do |_connection, config_fields|
        response_body = { name: 'body' }

        [
          if config_fields['response_type'] == 'raw'
            response_body
          elsif (output = config_fields['output'])
            output_schema = call('format_schema', parse_json(output))
            if output_schema.dig(0, 'type') == 'array' &&
               output_schema.dig(0, 'details', 'fake_array')
              response_body[:type] = 'array'
              response_body[:properties] = output_schema.dig(0, 'properties')
            else
              response_body[:type] = 'object'
              response_body[:properties] = output_schema
            end

            response_body
          end,
          if (headers = config_fields['response_headers'])
            header_props = parse_json(headers)&.map do |field|
              if field[:name].present?
                field[:name] = field[:name].gsub(/\W/, '_').downcase
              elsif field['name'].present?
                field['name'] = field['name'].gsub(/\W/, '_').downcase
              end
              field
            end

            { name: 'headers', type: 'object', properties: header_props }
          end
        ].compact
      end
    }
  },
  actions: {
    search_objects: {
      title: 'Search records',
      subtitle: 'Search records, e.g. person, in SalesLoft',
      description: lambda do |_input, pick_lists|
        "Search <span class='provider'>#{pick_lists[:object_name]&.
        downcase&.pluralize || 'objects'}</span> in <span class='provider'>"\
        'SalesLoft</span>'
      end,
      help: lambda do |_input, pick_lists|
        help = case pick_lists[:object_name]
               when 'User'
                 'Non Admin: Lists only your user, or all on team depending on group visibility' \
                 ' policy</br> Team Admin: Lists users associated with your team.'
               else
                 "Fetches multiple #{pick_lists[:object_name]&.downcase} " \
                 'records. The records can be filtered, paged and '\
                 'sorted according to the specified fields. By default, '\
                 'search result returns a maximum of <b>25</b> rows. Use the <b>Page'\
                 ' size</b> field to increase this limit up to <b>100</b> records.'
               end
        {
          body: help,
          learn_more_url: 'https://developers.salesloft.com/api.html#!/' \
          'Topic/Introduction',
          learn_more_text: 'SalesLoft API'
        }
      end,
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list_search',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do |object_definitions|
        object_definitions['object_search']
      end,
      execute: lambda do |_connection, input|
        query_params = input&.map do |key, val|
          if %w[ids crm_id tag name account_stage_id user_guid person_id
                owned_by_guid action_id email_template_id tag_ids user_ids
                email_addresses team_template_id crm_activity_id group_id
                person_stage_id guid].include?(key)
            { key => val&.split(',') }
          elsif key == 'updated_at'
            { "updated_at[#{val['operation']}" => val['date_updated'].
              to_time.utc.iso8601 }
          elsif key == 'created_at'
            { "created_at[#{val['operation']}" => val['date_created'].
              to_time.utc.iso8601 }
          elsif key == 'due_on'
            { "due_on[#{val['operation']}" => val['date_due'].
              to_time.utc.iso8601 }
          else
            { key => val }
          end
        end&.inject(:merge)

        call('run', 'fields' => query_params, 'action' => 'search')
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['record_search_output']
      end,
      sample_output: lambda do |_connection, input|
        {
          (input['object_name']&.pluralize || 'records') =>
          call('sample_record', input) || []
        }
      end
    },
    create_object: {
      title: 'Create record',
      subtitle: 'Create record, e.g. person, in SalesLoft',
      description: lambda do |_input, pick_lists|
        "Create <span class='provider'>#{pick_lists[:object_name]&.
        downcase || 'object'}</span> in <span class='provider'>"\
        'SalesLoft</span>'
      end,
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list_create',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do |object_definitions|
        object_definitions['object_create']
      end,
      execute: lambda do |_connection, input|
        query_params = input&.map do |key, val|
          if %w[linked_call_data_record_ids tags
                contact_restrictions].include?(key)
            { key => val&.split(',') }
          elsif key == 'view_params'
            { key => val.to_json }
          else
            { key => val }
          end
        end&.inject(:merge)
        call('run', 'fields' => query_params, 'action' => 'create')
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['object_output']
      end,
      sample_output: lambda do |_connection, input|
        call('sample_record', input)
      end
    },
    get_object: {
      title: 'Get record',
      subtitle: 'Get record details, e.g. person, in SalesLoft',
      description: lambda do |_input, pick_lists|
        "Get <span class='provider'>#{pick_lists[:object_name]&.
        downcase || 'object'}</span> by ID in <span class='provider'>"\
        'SalesLoft</span>'
      end,
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list_get',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do |object_definitions|
        object_definitions['object_get']
      end,
      execute: lambda do |_connection, input|
        call('run', 'fields' => input, 'action' => 'get')
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['object_output']
      end,
      sample_output: lambda do |_connection, input|
        call('sample_record', input)
      end
    },
    update_object: {
      title: 'Update record',
      subtitle: 'Update record, e.g. person, in SalesLoft',
      description: lambda do |_input, pick_lists|
        "Update <span class='provider'>#{pick_lists[:object_name]&.
        downcase || 'object'}</span> in <span class='provider'>"\
        'SalesLoft</span>'
      end,
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list_update',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do |object_definitions|
        object_definitions['object_update']
      end,
      execute: lambda do |_connection, input|
        query_params = input&.map do |key, val|
          if %w[tags contact_restrictions].include?(key)
            { key => val&.split(',') }
          elsif key == 'view_params'
            { key => val.to_json }
          else
            { key => val }
          end
        end&.inject(:merge)
        call('run', 'fields' => query_params, 'action' => 'update')
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['object_output']
      end,
      sample_output: lambda do |_connection, input|
        call('sample_record', input)
      end
    },
    delete_object: {
      title: 'Delete record',
      subtitle: 'Delete record, e.g. person, in SalesLoft',
      description: lambda do |_input, pick_lists|
        "Delete <span class='provider'>#{pick_lists[:object_name]&.
        downcase || 'object'}</span> in <span class='provider'>"\
        'SalesLoft</span>'
      end,
      help: {
        body: 'Deletes a record of a selected object.<br/><b>Note: </b>'\
        'This process is not reversible.',
        learn_more_url: 'https://developers.salesloft.com/api.html#!/' \
        'Topic/Introduction',
        learn_more_text: 'SalesLoft API'
      },
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list_delete',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do |object_definitions|
        object_definitions['object_get']
      end,
      execute: lambda do |_connection, input|
        call('run', 'fields' => input, 'action' => 'delete')
      end,
      output_fields: lambda do |_object_definitions|
        [{ name: 'status' }]
      end,
      sample_output: lambda do |_connection, _input|
        { 'status': 'Success' }
      end
    },
    custom_action: {
      subtitle: 'Build your own SalesLoft action with a HTTP request',

      description: lambda do |object_value, _object_label|
        "<span class='provider'>" \
        "#{object_value[:action_name] || 'Custom action'}</span> in " \
        "<span class='provider'>SalesLoft</span>"
      end,

      help: {
        body: 'Build your own SalesLoft action with a HTTP request. ' \
        'The request will be authorized with your SalesLoft connection.',
        learn_more_url: 'https://developers.salesloft.com/api.html',
        learn_more_text: 'SalesLoft API documentation'
      },

      config_fields: [
        {
          name: 'action_name',
          hint: "Give this action you're building a descriptive name, e.g. " \
          'create record, get record',
          default: 'Custom action',
          optional: false,
          schema_neutral: true
        },
        {
          name: 'verb',
          label: 'Method',
          hint: 'Select HTTP method of the request',
          optional: false,
          control_type: 'select',
          pick_list: %w[get post put patch options delete].
            map { |verb| [verb.upcase, verb] }
        }
      ],

      input_fields: lambda do |object_definition|
        object_definition['custom_action_input']
      end,

      execute: lambda do |_connection, input|
        verb = input['verb']
        if %w[get post put patch options delete].exclude?(verb)
          error("#{verb.upcase} not supported")
        end
        path = input['path']
        data = input.dig('input', 'data') || {}
        if input['request_type'] == 'multipart'
          data = data.each_with_object({}) do |(key, val), hash|
            hash[key] = if val.is_a?(Hash)
                          [val[:file_content],
                           val[:content_type],
                           val[:original_filename]]
                        else
                          val
                        end
          end
        end
        request_headers = input['request_headers']&.
          each_with_object({}) do |item, hash|
          hash[item['key']] = item['value']
        end || {}
        request = case verb
                  when 'get'
                    get(path, data)
                  when 'post'
                    if input['request_type'] == 'raw'
                      post(path).request_body(data)
                    else
                      post(path, data)
                    end
                  when 'put'
                    if input['request_type'] == 'raw'
                      put(path).request_body(data)
                    else
                      put(path, data)
                    end
                  when 'patch'
                    if input['request_type'] == 'raw'
                      patch(path).request_body(data)
                    else
                      patch(path, data)
                    end
                  when 'options'
                    options(path, data)
                  when 'delete'
                    delete(path, data)
                  end.headers(request_headers)
        request = case input['request_type']
                  when 'url_encoded_form'
                    request.request_format_www_form_urlencoded
                  when 'multipart'
                    request.request_format_multipart_form
                  else
                    request
                  end
        response =
          if input['response_type'] == 'raw'
            request.response_format_raw
          else
            request
          end.
          after_error_response(/.*/) do |code, body, headers, message|
            error({ code: code, message: message, body: body, headers: headers }.
            to_json)
          end

        response.after_response do |_code, res_body, res_headers|
          {
            body: res_body ? call('format_response', res_body) : nil,
            headers: res_headers
          }
        end
      end,

      output_fields: lambda do |object_definition|
        object_definition['custom_action_output']
      end
    }
  },
  triggers: {
    new_updated_object: {
      title: 'New/updated record',
      subtitle: 'Triggers when SalesLoft object, e.g. person, is created/updated',
      description: lambda do |_input, pick_lists|
        "New/updated <span class='provider'>#{pick_lists[:object_name]&.
      downcase || 'object'} </span> in <span class='provider'>SalesLoft</span>"
      end,
      help: {
        body: 'Triggers for every created/updated object. Checks for new ' \
        'events every 5 minutes.',
        learn_more_url: 'https://developers.salesloft.com/api.html#!/Topic/'\
        'Introduction',
        learn_more_text: 'SalesLoft API'
      },
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do
        [{ name: 'since', type: 'date_time', optional: true, sticky: true,
           label: 'When first started, this recipe should pick up events from',
           hint: 'Leave empty to get events created one hour ago' }]
      end,
      poll: lambda do |_connection, input, closure|
        page_size = 100
        page = closure&.[]('page') || 1
        last_updated_since = closure&.[]('last_modified_datetime') ||
                             (input['since'].presence || 1.hour.ago).
                             to_time.utc.iso8601
        path = if %w[calls emails].include?(input['object_name'])
                 "activities/#{input['object_name']}"
               else
                 input['object_name']
               end
        response = get("v2/#{path}.json").
                   params(sort_by: 'updated_at',
                          sort_direction: 'ASC',
                          per_page: page_size,
                          page: page,
                          'updated_at[gte]': last_updated_since,
                          include_paging_counts: true)
        object = response['data']
        has_more_pages = response&.
                          dig('metadata', 'paging', 'next_page').present?
        closure = if has_more_pages
                    { 'last_modified_datetime' => last_updated_since,
                      'page' => page + 1 }
                  else
                    last_modified_date = object&.dig(-1, 'updated_at') ||
                                           now.to_time.utc.iso8601
                    { 'last_modified_datetime' => last_modified_date,
                      'page' => 1 }
                  end

        {
          events: object,
          next_poll: closure,
          can_poll_more: has_more_pages
        }
      end,

      dedup: lambda do |object|
        "#{object['id']}@#{object['updated_at']}"
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['object_output']
      end,
      sample_output: lambda do |_connection, input|
        path = if %w[calls emails].include?(input['object_name'])
                 "activities/#{input.delete('object_name')}"
               else
                 input.delete('object_name')
               end
        get("v2/#{path}.json?per_page=1")&.dig('data', 0) || {}
      end
    },
    new_object: {
      title: 'New record',
      subtitle: 'Triggers when SalesLoft object, e.g. person, is created',
      description: lambda do |_input, pick_lists|
        "New <span class='provider'>#{pick_lists[:object_name]&.
      downcase || 'object'} </span> in <span class='provider'>SalesLoft</span>"
      end,
      help: {
        body: 'Triggers for newly created object. Checks for new ' \
        'events every 5 minutes.',
        learn_more_url: 'https://developers.salesloft.com/api.html#!/Topic/'\
        'Introduction',
        learn_more_text: 'SalesLoft API'
      },
      config_fields: [
        { name: 'object_name', control_type: 'select',
          pick_list: 'object_list_new',
          label: 'Object name',
          optional: false }
      ],
      input_fields: lambda do
        [{ name: 'since', type: 'date_time', optional: true, sticky: true,
           label: 'When first started, this recipe should pick up events from',
           hint: 'Leave empty to get events created one hour ago' }]
      end,
      poll: lambda do |_connection, input, closure|
        page_size = 100
        page = closure&.[]('page') || 1
        last_updated_since = closure&.[]('last_modified_datetime') ||
                             (input['since'].presence || 1.hour.ago).
                             to_time.utc.iso8601
        response = get("v2/#{input['object_name']}.json").
                   params(sort_by: 'created_at',
                          sort_direction: 'ASC',
                          per_page: page_size,
                          page: page,
                          'created_at[gte]': last_updated_since,
                          include_paging_counts: true)
        object = response['data']
        has_more_pages = response&.
                          dig('metadata', 'paging', 'next_page').present?
        closure = if has_more_pages
                    { 'last_modified_datetime' => last_updated_since,
                      'page' => page + 1 }
                  else
                    last_modified_date = object&.dig(-1, 'created_at') ||
                                           now.to_time.utc.iso8601
                    { 'last_modified_datetime' => last_modified_date,
                      'page' => 1 }
                  end

        {
          events: object,
          next_poll: closure,
          can_poll_more: has_more_pages
        }
      end,

      dedup: lambda do |object|
        "#{object['id']}@#{object['created_at']}"
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['object_output']
      end,
      sample_output: lambda do |_connection, input|
        get("v2/#{input['object_name']}.json?per_page=1")&.dig('data', 0) || {}
      end
    }
  },
  pick_lists: {
    cadences: lambda do |_connection|
      get('v2/cadences.json')['data']&.pluck('name', 'id')
    end,
    person_stages: lambda do |_connection|
      get('v2/person_stages.json')['data']&.pluck('name', 'id')
    end,
    imports: lambda do |_connection|
      get('v2/imports.json')['data']&.pluck('name', 'id')
    end,
    owners: lambda do |_connection|
      get('v2/users.json')['data']&.pluck('name', 'id')
    end,
    operators: lambda do |_connection|
      [
        ['Greater than', 'gt'],
        ['Greater than or equal', 'gte'],
        ['Less than', 'lt'],
        ['Less than or equal', 'lte']
      ]
    end,
    job_seniorities: lambda do |_connection|
      %w[director executive individual_contributor manager
         vice_president unknown].map { |option| [option.labelize, option] }
    end,
    contact_restrictions: lambda do |_connection|
      %w[call email message].
        map { |option| [option.labelize, option] }
    end,
    object_list: lambda do |_connection|
      [
        %w[Cadence cadences],
        %w[Cadence\ membership cadence_memberships],
        %w[Person people],
        %w[Call calls],
        %w[Email emails],
        %w[CRM\ activity crm_activities],
        %w[Call\ data\ record call_data_records],
        %w[Account accounts],
        %w[Note notes],
        %w[Account\ stage account_stages],
        %w[Email\ template email_templates],
        %w[Success successes],
        %w[Team\ templates team_templates]
      ]
    end,
    object_list_new: lambda do |_connection|
      [
        %w[Cadence cadences],
        %w[Person people]
      ]
    end,
    object_list_search: lambda do |_connection|
      [
        %w[Cadence cadences],
        %w[Cadence\ membership cadence_memberships],
        %w[Person people],
        %w[Call calls],
        %w[Email emails],
        %w[CRM\ activity crm_activities],
        %w[Call\ data\ record call_data_records],
        %w[Account accounts],
        %w[Note notes],
        %w[Person\ stage person_stages],
        %w[Saved\ list\ view saved_list_views],
        %w[Account\ stage account_stages],
        %w[Account\ tier account_tiers],
        %w[Action actions],
        %w[Call\ disposition call_dispositions],
        %w[Call\ sentiment call_sentiments],
        %w[Call\ instruction call_instructions],
        %w[Custom\ field custom_fields],
        %w[Email\ template\ attachment email_template_attachments],
        %w[Email\ template email_templates],
        %w[Import imports],
        %w[Step steps],
        %w[Success successes],
        %w[Team\ tag tags],
        %w[Team\ template\ attachment team_template_attachments],
        %w[Team\ template team_templates],
        %w[User users]
      ]
    end,
    object_list_create: lambda do |_connection|
      [
        %w[Account accounts],
        %w[Activity activities],
        %w[Person\ to\ cadence cadence_memberships],
        %w[Call calls],
        %w[Custom\ field custom_fields],
        %w[Import imports],
        %w[Note notes],
        # %w[Ongoing\ action ongoing_actions],
        %w[Person people],
        %w[Person\ stage person_stages],
        %w[Saved\ list\ view saved_list_views]
      ]
    end,
    object_list_update: lambda do |_connection|
      [
        %w[Account accounts],
        %w[Custom\ field custom_fields],
        %w[Import imports],
        %w[Note notes],
        %w[Person people],
        %w[Person\ stage person_stages],
        %w[Saved\ list\ view saved_list_views]
      ]
    end,
    object_list_delete: lambda do |_connection|
      [
        %w[Account accounts],
        %w[Person\ from\ cadence cadence_memberships],
        %w[Custom\ field custom_fields],
        %w[Import imports],
        %w[Note notes],
        %w[Person people],
        %w[Person\ stage person_stages],
        %w[Saved\ list\ view saved_list_views]
      ]
    end,
    object_list_get: lambda do |_connection|
      [
        %w[Account accounts],
        %w[Account\ stage account_stages],
        %w[Account\ tier account_tiers],
        %w[Action actions],
        %w[Call\ instruction call_instructions],
        %w[Cadence cadences],
        %w[Cadence\ membership cadence_memberships],
        %w[Call\ data\ record call_data_records],
        %w[Call calls],
        %w[CRM\ activity crm_activities],
        %w[Custom\ field custom_fields],
        %w[Email emails],
        %w[Email\ template email_templates],
        %w[Import imports],
        %w[Note notes],
        %w[Person people],
        %w[Person\ stage person_stages],
        %w[Saved\ list\ view saved_list_views],
        %w[Step steps],
        %w[Team\ template team_templates],
        %w[User users]
      ]
    end,
    sort_fields: lambda do |_connection, object:|
      {
        'accounts' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at],
          %w[Last\ contacted\ at last_contacted_at],
          %w[Account\ stage account_stage],
          %w[Account\ tier account_tier]
        ],
        'account_stages' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'account_tiers' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'actions' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'call_instructions' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'cadences' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'cadence_memberships' => [
          %w[Added\ date added_at],
          %w[Updated\ date updated_at]
        ],
        'call_data_records' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'call_dispositions' => [
          %w[Name name],
          %w[Updated\ date updated_at]
        ],
        'calls' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'call_sentiments' => [
          %w[Name name],
          %w[Updated\ date updated_at]
        ],
        'crm_activities' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'custom_fields' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at],
          %w[Name name]
        ],
        'emails' => [
          %w[Updated\ date updated_at],
          %w[Send\ time send_time]
        ],
        'email_templates' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at],
          %w[Last\ used\ date last_used_at]
        ],
        'imports' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'notes' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'people' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at],
          %w[Last\ contacted\ date last_contacted_at]
        ],
        'person_stages' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'saved_list_views' => [
          %w[Name name]
        ],
        'steps' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at]
        ],
        'successes' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at],
          %w[Succeeded\ date succeeded_at]
        ],
        'tags' => [
          %w[Name name]
        ],
        'team_templates' => [
          %w[Created\ date created_at],
          %w[Updated\ date updated_at],
          %w[Last\ used\ date last_used_at]
        ]
      }[object]
    end,
    sort_directions: lambda do |_connection|
      [
        %w[Ascending ASC],
        %w[Descending DESC]
      ]
    end,
    associated_types: lambda do |_connection|
      [
        %w[Person person],
        %w[Account account]
      ]
    end,
    field_types: lambda do |_connection|
      [
        %w[Person person],
        %w[Company company],
        %w[Opportunity opportunity]
      ]
    end,
    view_types: lambda do |_connection|
      [
        %w[People people],
        %w[Companies companies]
      ]
    end,
    step_types: lambda do |_connection|
      [
        %w[Email email],
        %w[Phone phone],
        %w[Integration integration],
        %w[Other other]
      ]
    end,
    call_dispositions: lambda do |_connection|
      get('/v2/call_dispositions.json')['data']&.pluck('name', 'name')
    end,
    call_sentiments: lambda do |_connection|
      get('/v2/call_sentiments.json')['data']&.pluck('name', 'name')
    end
  }
}
